<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models
* @filesource
*/

/** */
require_once 'folder.php';
require_once 'message.php';
require_once 'user.php';

/**
* @author M. Gibbs <gibbs_margaret@bah.com>
* @package direct-as-a-service
* @subpackage models
*/
class Mailbox extends Entity {
	static $database_group = 'mail_db';
	static $table = 'mailboxes';	
	
	protected $_facility;
	
	/** 
	* The configuration for this model's relationships.
	*
	* Entities make use of PHP's magical methods ({@link __get}, {@link __ set}, and {@__call} to create virtual variables and methods that will search for
	* related entities, count the related entities, and provide a boolean true/false as to whether there are any entities related to this entity via this
	* relationship.  For example, for the inbox_message relationship, we have all of these variables and methods available just by defining the relationship:
	*
	* <code>
	* $mailbox = Mailbox::find_one( array('username' => jswift') );
	* $mailbox->inbox_messages; //an array of all the messages in this mailbox's inbox 
	* $mailbox->inbox_messages( array( 'subject' => 'A Modest Proposal' )); //an array of all the messages in this mailbox's inbox that match the search criteria
	* $mailbox->inbox_message_count; //the number of messages in the inbox
	* $mailbox->inbox_message_count( array( 'subject' => 'A Modest Proposal' )); //number of messages that match the search criteria
	* $mailbox->has_inbox_messages; //true if there are any inbox messages
	* $mailbox->has_inbox_messages( array( 'subject' => 'A Modest Proposal' )); //true if there are messages that match the search criteria
	* </code>
	*
	* Note that the Mailbox class additionally allows the developer to append a message part (see {@link Message}) to the end of a message relationship to 
	* limit the selection just the fields relevant to the part.
	*
	* <code>
	* $mailbox->inbox_message_headers; //only the header fields will be selected for the inbox messages
	* $mailbox->inbox_message_headers( array('timestamp < ' => mktime() ) ); //only the header fields will be selected for the search results
	* </code>
	*
	* This default behavior will be overriden if you define any methods/variables by these names for this class.  Also note that since these are not *actually* 
	* variables on the class, isset() will always return false.  To see the details of how the values for these variables/methods are determined, see 
	* {@link __get} and {@link __call}.
	*
	* For more information on how to set up or configure relationships, see {@link relationships()}. 
	*
	* @var array
	*/
	protected static $_relationships = array( 'folder' => array('type' => 'has_many'),
											  'message' => array('type' => 'has_many'),
											  'inbox_message' => array( 'type' => 'has_many', 'model' => 'message',  'condition' => 'folder_id IS NULL AND sent=0 AND draft=0 AND archived=0'),
											  'archived_message' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'archived=1'),
											  'sent_message' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'sent=1 AND archived=0'),
											  'draft' => array( 'type' => 'has_many', 'model' => 'message', 'condition' => 'draft=1 AND archived=0'),
											  'user' => array( 'type' => 'belongs_to', 'related_foreign_key' => 'name', 'key_for_relationship' => 'username' ),
											 );
											 
	protected $_property_validation_rules = array( 'name' => 'nonempty_string',
												   'is_group' => 'boolean',
												   'is_active' => 'boolean',
												   'facility_id' => 'null|nonzero_unsigned_integer' );												 
	
	
//////////////////////////////////////////////////////////////////////////////////////////
// INSTANCE METHODS 
// These methods assume that the instance represents a mailbox record in the database.
//////////////////////////////////////////////////////////////////////////////////////////			
	
	function add_folder($values){
		if(!$this->is_active) return $this->error->warning("I can't add a new folder to ".$this->describe().' while it is disabled');
		$foreign_key = static::foreign_key('folder');
		$key_for_relationship = static::key_for_relationship('folder');
		$values = array_merge($values, array($foreign_key => $this->$key_for_relationship));
		return Folder::create($values);
	}		
	
	function create_message($values){
		if(!$this->is_active) return $this->error->warning("I can't create a new message for ".$this->describe().' while it is inactive');
		if(!isset($this->id)) return $this->error->warning("I can't create a new message for ".$this->describe().' before it is saved to the database');
		
		$foreign_key = static::foreign_key('message');
		$values[$foreign_key] = $this->id;
		return Message::create($values);
	}

	function dn(){
		if($this->is_group){
			if(isset($this->id) && !$this->is_active) return get_instance()->groupsmodel->get_disabled_dn_from_groupname($this->name);
			return get_instance()->groupsmodel->get_dn_from_groupname($this->name);
		}
		if(isset($this->id) && !$this->is_active) return get_instance()->usersmodel->get_disabled_dn_from_username($this->name);
		return get_instance()->usersmodel->get_dn_from_username($this->name);
	}											 

	function facility($facility_field = NULL){
		if(isset($this->_facility) && $this->_facility['id'] != $this->facility_id) $this->_facility_id = null;
		if(!isset($this->_facility) && !$this->property_is_empty('facility_id')){
			$query = get_instance()->facilitymodel->get_facility($this->facility_id);
			if($query->num_rows() > 0){
				$this->_facility = $query->row();
			}
		}
		
		if(!is_null($facility_field) && is_object($this->_facility)){
			return $this->_facility->$facility_field;
		}		
		
		return $this->_facility;
	}

	/** 
	* The email address for this mailbox.
	* @return string
	*/
	public function email_address(){
		if($this->property_is_empty('name')) return '';
		return $this->name.'@'.DIRECT_DOMAIN;
	}									 	

	function locations(){
		return array('inbox'=>'Inbox','sent'=>'Sent','draft'=>'Drafts','archived'=>'Archived') + collect('name', $mailbox->folders);
	}	
	
	/**
	* Statics on sent/received messagse for this mailbox & group mailboxes that the mailbox's owner belongs to.
	* This method only tracks sent/received messages - draft messages are not included.
	* @return array
	* @return array 
	* @return array
	*/
	function message_activity_statistics($conditions=array(), $group_mailboxes_to_include = array()){
		if(!is_array($conditions)) return $this->error->should_be_an_array_of_db_conditions($conditions);		
		if(!$this->is->array_of_nonempty_strings($group_mailboxes_to_include)) return $this->error->should_be_an_array_of_group_mailbox_names($group_mailboxes_to_include);
#TODO - HOW TO CHECK MEMBERSHIP FOR A USER?		
		if($this->is_group && !empty($group_mailboxes_to_include)) return $this->error->should_be_a_list_of_group_mailboxes_this_user_belongs_to($group_mailboxes_to_include);
			
		//find the mailboxes so that we can make sure that they're group mailboxes
		$group_mailboxes = array();
		if(!empty($group_mailboxes_to_include)){
			Mailbox::db()->where_in('name', $group_mailboxes_to_include);
			$group_mailboxes = Mailbox::find( array('is_group' => true));	
		}	
		$mailbox_ids = array_keys($group_mailboxes);
		array_unshift($mailbox_ids, $this->id);

		//find the counts for all of the mailboxes at once so that we can avoid excess db queries
		$counts = array();
		$conditions_for_counts = array( 'sent_total' => array_merge($conditions, array('sent' => true)),
										'sent_by_you' => array_merge($conditions, array('sent' => true, 'original_sender_id' => $this->id)),
										'received' => array_merge($conditions, array('sent' => false, 'draft' => false)),
										'total' => array_merge($conditions, array('draft' => false)) );
										
		foreach($conditions_for_counts as $count_type => $count_conditions){
			Message::db()->where_in('mailbox_id', $mailbox_ids);
			$counts[$count_type] = Message::count_by('mailbox_id', $count_conditions);		
		}										
		
		//separate them out by mailbox to make it easier to view
		$counts_by_mailbox = array();
		$i = 1;
		foreach($mailbox_ids as $mailbox_id){
			$mailbox = element($mailbox_id, $group_mailboxes, $this);
			$counts_by_mailbox[$mailbox->email_address] = array( 'default_order' => $i,
																 'address' => $mailbox->email_address,
																 'sent_by_you' => element($mailbox->id, $counts['sent_by_you'], 0),
																 'sent_total' => element($mailbox->id, $counts['sent_total'], 0),
																 'received' => element($mailbox->id, $counts['received'], 0),
																 'total' => element($mailbox->id, $counts['total'], 0),
																 );
																 
			$i++;
		}
			
		return $counts_by_mailbox;		
	}	
		
///////////////////////////////////////
// GETTERS
// For class vars prefixed with a _
///////////////////	///////////////////
	
	public function readonly_fields(){
		$readonly_fields = $this->_merge_with_parent_array('readonly_fields');
		//if the mailbox isn't active, all fields except for is_active are readonly
		if(isset($this->id) && !$this->is_active){
			$readonly_fields = array_unique( array_merge( $readonly_fields, array_diff( static::fields(), array('is_active') ) ) );
		}
		return $readonly_fields;
    }
	
	

	function __get($property){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		
		//for message relationships, allow developers to append the name of a message part to the relationship name
		//e.g. $this->inbox_message_headers will return the headers for $this->inbox_messages
		$suffix = mb_substr($property, mb_strrpos($property, '_') + 1);
		if(Message::is_a_part($suffix)){
			$relationship = strip_from_end('_'.$suffix, $property);
			if($class::has_relationship($relationship) && $class::relationships($relationship, 'model') == 'message'){
				$this->set_up_relationship_query($relationship);
				return Message::find_part($suffix);
			}
		}
				
		return parent::__get($property);
	}
	
	function __call($name, $arguments){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		
		//for message relationships, allow developers to append the name of a message part to the relationship name
		//e.g. $this->inbox_message_headers() will return the headers for $this->inbox_messages()
		$suffix = mb_substr($name, mb_strrpos($name, '_') + 1);
		if(Message::is_a_part($suffix)){
			$relationship = strip_from_end('_'.$suffix, $name);
			if($class::has_relationship($relationship) && $class::relationships($relationship, 'model') == 'message'){
				$this->set_up_relationship_query($relationship);
				return call_user_func_array('Message::find_part', array_merge( array($suffix), $arguments));
			}
		}
				
		return parent::__call($name, $arguments);
	}	
	
/////////////////////
// STATIC FUNCTIONS
/////////////////////
	
	public static function find_by_email_address($address){
		if(!get_instance()->is->string_like_a_direct_address($address)) return get_instance()->error->should_be_a_direct_address($address);
		$name = strip_from_end('@'.DIRECT_DOMAIN, $address);
		return static::find_one( compact('name') );
	}
	


	
}
